import { Request, Response } from 'express';
import { referralsService } from './service';

function hasFeature(user: any, featureKey: string): boolean {
  return user?.features?.includes(featureKey);
}

export const referralsController = {
  // POST /referrals
  async register(req: any, res: Response) {
    const user = req.user;
    if (!user || !hasFeature(user, 'referrals')) {
      return res.status(403).json({ error: 'Referrals not available on your plan.' });
    }
    const { data, error } = await referralsService.registerReferral(user.id, req.body);
    if (error) return res.status(400).json({ error: error.message });
    return res.status(201).json({ success: true, data });
  },

  // GET /referrals
  async list(req: any, res: Response) {
    const user = req.user;
    if (!user || !hasFeature(user, 'referrals')) {
      return res.status(403).json({ error: 'Referrals not available on your plan.' });
    }
    const { data, error } = await referralsService.listReferrals(user.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // GET /referrals/summary
  async summary(req: any, res: Response) {
    const user = req.user;
    if (!user || !hasFeature(user, 'referrals')) {
      return res.status(403).json({ error: 'Referrals not available on your plan.' });
    }
    const { data, error } = await referralsService.getSummary(user.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  },

  // PATCH /referrals/admin/:id
  async adminReview(req: any, res: Response) {
    const admin = req.admin;
    if (!admin) return res.status(401).json({ error: 'Unauthorized' });
    const { id } = req.params;
    const { data, error } = await referralsService.adminReviewReferral(id, req.body, admin.id);
    if (error) return res.status(400).json({ error: error.message });
    return res.json({ success: true, data });
  }
}; 